#pragma once

#ifdef __cplusplus

// use if you have memory allocated and want copy image data from an IImageSource
// example: pYourImageSource->DataGet(CImageSinkOnMemory(pYourAllocatdMemory, nYourAllocatdMemoryLength));
// note: COM AddRef/Release are not correctly implemented to simplify the use scenario
class CImageSinkOnMemory :
	public IImageSink
{
public:
	CImageSinkOnMemory(BYTE* a_pData, ULONG a_nLen) : m_pData(a_pData), m_nLen(a_nLen)
	{
	}

	operator IImageSink*()
	{
		return this;
	}

	STDMETHOD_(ULONG, AddRef)()
	{
		return 1;
	}
	STDMETHOD_(ULONG, Release)()
	{
		return 1;
	}
	STDMETHOD(QueryInterface)(REFIID a_guidIID, void** a_ppInterface)
	{
		if (IsEqualIID(a_guidIID, IID_IUnknown) ||
			IsEqualIID(a_guidIID, __uuidof(IImageSink)))
		{
			*a_ppInterface = (void*)this;
			return S_OK;
		}
		return E_NOINTERFACE;
	}

	STDMETHOD(BufferLock)(ULONG a_nOffset, DWORD a_nLength, BYTE** a_ppBuffer)
	{
		if ((a_nOffset+a_nLength) > m_nLen)
		{
			return E_FAIL;
		}

		*a_ppBuffer = m_pData+a_nOffset;

		return S_OK;
	}
	STDMETHOD(BufferUnlock)(DWORD /*a_nLength*/, BYTE* /*a_pBuffer*/)
	{
		return S_OK;
	}

private:
	BYTE* m_pData;
	ULONG m_nLen;
};

class CImageSinkOnMemoryAutoAlloc :
	public IImageSink
{
public:
	CImageSinkOnMemoryAutoAlloc() : m_pData(NULL), m_nLen(0)
	{
	}
	~CImageSinkOnMemoryAutoAlloc()
	{
		delete[] m_pData;
	}

	operator IImageSink*()
	{
		return this;
	}

	ULONG Length() const
	{
		return m_nLen;
	}

	BYTE const* Data() const
	{
		return m_pData;
	}

	BYTE* Detach()
	{
		BYTE* pData = m_pData;
		m_pData = NULL;
		m_nLen = 0;
		return pData;
	}


	STDMETHOD_(ULONG, AddRef)()
	{
		return 1;
	}
	STDMETHOD_(ULONG, Release)()
	{
		return 1;
	}
	STDMETHOD(QueryInterface)(REFIID a_guidIID, void** a_ppInterface)
	{
		if (IsEqualIID(a_guidIID, IID_IUnknown) ||
			IsEqualIID(a_guidIID, __uuidof(IImageSink)))
		{
			*a_ppInterface = (void*)this;
			return S_OK;
		}
		return E_NOINTERFACE;
	}

	STDMETHOD(BufferLock)(ULONG a_nOffset, DWORD a_nLength, BYTE** a_ppBuffer)
	{
		if ((a_nOffset+a_nLength) > m_nLen)
		{
			try
			{
				BYTE* pData = new BYTE[a_nOffset+a_nLength];
				memcpy(pData, m_pData, m_nLen);
				delete[] m_pData;
				m_pData = pData;
				m_nLen = a_nOffset+a_nLength;
			}
			catch (...)
			{
				return E_FAIL;
			}
		}

		*a_ppBuffer = m_pData+a_nOffset;

		return S_OK;
	}
	STDMETHOD(BufferUnlock)(DWORD /*a_nLength*/, BYTE* /*a_pBuffer*/)
	{
		return S_OK;
	}

private:
	BYTE* m_pData;
	ULONG m_nLen;
};


// derive your class from CImageSourceSimple intead of IImageSource
// to automatically implement PullISGet method
class CImageSourceSimple :
	public IImageSource
{
public:
	CImageSourceSimple() : m_cISP(this)
	{
	}

	STDMETHOD(PullISGet)(IImageSourcePull** a_ppISSource)
	{
		return m_cISP.QueryInterface(__uuidof(IImageSourcePull), reinterpret_cast<void**>(a_ppISSource));
	}

public: // TODO: make it protected once IconImage is fixed
	void DeleteCachedData()
	{
		m_cISP.DeleteCachedData();
	}

private:
	class CImageSourcePullOnIS :
		public IImageSourcePull
	{
	public:
		CImageSourcePullOnIS(IImageSource* a_pHost) : m_pHost(a_pHost), m_pData(NULL), m_hRes(E_FAIL)
		{
			ATLASSERT(a_pHost);
		}
		~CImageSourcePullOnIS()
		{
			delete[] m_pData;
		}

		// IUnknown methods
	public:
		STDMETHOD_(ULONG, AddRef)()
		{
			return m_pHost->AddRef();
		}
		STDMETHOD_(ULONG, Release)()
		{
			return m_pHost->Release();
		}
		STDMETHOD(QueryInterface)(REFIID a_iid, void** a_ppOut)
		{
			if (a_ppOut == NULL)
				return E_POINTER;

			if (IsEqualIID(a_iid, __uuidof(IImageSourcePull)) ||
				IsEqualIID(a_iid, IID_IUnknown))
			{
				*a_ppOut = reinterpret_cast<void*>(this);
				m_pHost->AddRef();
				return S_OK;
			}
			else
			{
				*a_ppOut = NULL;
				return E_NOINTERFACE;
			}
		}

		// IImageSourcePull methods
	public:
		STDMETHOD(BufferLock)(ULONG a_nOffset, DWORD /*a_nLength*/, BYTE const** a_ppBuffer)
		{
			if (m_pData == NULL)
			{
				TImageFormat tIF;
				m_pHost->FormatGet(&tIF);
				m_nSize = tIF.nDataSize;
				if (m_nSize != 0)
				{
					m_pData = new BYTE[m_nSize];
					m_hRes = m_pHost->DataGet(CImageSinkOnMemory(m_pData, m_nSize));
				}
				else
				{
					CImageSinkOnMemoryAutoAlloc cAutoAllocSink;
					m_hRes = m_pHost->DataGet(cAutoAllocSink);
					m_nSize = cAutoAllocSink.Length();
					m_pData = cAutoAllocSink.Detach();
				}
			}
			*a_ppBuffer = m_pData+a_nOffset;
			return m_hRes;
		}
		STDMETHOD(BufferUnlock)(DWORD /*a_nLength*/, BYTE const* /*a_pBuffer*/)
		{
			return S_OK;
		}

	public:
		void DeleteCachedData()
		{
			delete[] m_pData;
			m_pData = NULL;
		}

	private:
		IImageSource* m_pHost;
		BYTE* m_pData;
		DWORD m_nSize;
		HRESULT m_hRes;
	};

private:
	CImageSourcePullOnIS m_cISP;
};

#endif//__cplusplus

